<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Mail;

use Illuminate\Http\Request;
use App\Models\custom_field;
use App\Models\visitor;
use Carbon\Carbon;

use App\Models\notification;
use App\Models\visit;
use App\Models\activity_log;
use App\Mail\invitation_mail;
use App\Models\visitor_custom_field_value;
use Illuminate\Support\Str;
use App\Http\Classes\Generic;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\company_user;
use App\Models\visit_host;
use App\Models\visit_host_response;
use App\Http\Traits\SystemMail;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\DB;
class VisitController extends Controller
{
      use SystemMail;
public function store(Request $req)
{
    // return $req;
    if ($req->isMethod('post')) {

        $validator = Validator::make($req->all(), [
            'firstName' => 'required|string|max:255',
            'lastName' => 'required|string|max:255',
            'company' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'idPhoto' => 'required|string',
            'purpose' => 'required|string|max:150',
            'customFields' => 'array',
            'customFields.*.id' => 'required|integer',
            'schedule' => 'required|string'
        ]);

        if ($validator->fails()) {
            if ($req->ajax()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        try {
            return DB::transaction(function () use ($req) {

                $v_host = "";
                $visitor = visitor::where('email', $req->email)->first();

                if ($visitor) {
                    $visitor->first_name = $req->firstName;
                    $visitor->last_name = $req->lastName;
                    $visitor->organization = $req->company;
                    $visitor->phone = $req->phone;
                    $visitor->whatsapp = $req->whatsapp ?? $visitor->whatsapp;
                    $visitor->city = $req->city;
                    $visitor->state = $req->state;
                    $visitor->country = $req->country;
                    $visitor->address = $req->address ?? $visitor->address;
                    $visitor->blacklisted = "false";
                } else {
                    $visitor = new visitor();
                    $visitor->external_id = strtoupper(Str::random(6));
                    $visitor->first_name = $req->firstName;
                    $visitor->last_name = $req->lastName;
                    $visitor->company_id = session('company_id');
                    $visitor->organization = $req->company;
                    $visitor->email = $req->email;
                    $visitor->phone = $req->phone;
                    $visitor->whatsapp = $req->whatsapp ?? null;
                    $visitor->city = $req->city;
                    $visitor->state = $req->state;
                    $visitor->country = $req->country;
                    $visitor->blacklisted = "false";
                    $visitor->address = $req->address ?? null;
                }

                if ($req->idPhoto) {
                    $imageData = str_replace('data:image/png;base64,', '', $req->idPhoto);
                    $imageData = str_replace(' ', '+', $imageData);
                    $imageName = 'visitor_' . time() . '.png';
                    $imagePath = storage_path('app/public/visitors/' . $imageName);
                    file_put_contents($imagePath, base64_decode($imageData));
                    $visitor->photo_path = 'visitors/' . $imageName;
                }

                $visitor->purpose_of_visit = $req->purpose;
                $visitor->save();

                $visit = new visit();
                $visit->visitor_id = $visitor->id;
                $visit->external_id = strtoupper(Str::random(6));
                $visit->department_id = $req->departmentId;
                $visit->visit_type = $req->visitType;
                $visit->purpose = $req->purpose;
                $visit->status = "requested";
                $visit->checkin_time = now();
                $visit->visit_date = now();
                $visit->company_id = session('company_id');
                $visit->scheduled = $req->schedule ?? 'Unscheduled';
                $visit->save();

                $hostsData = [];

                if ($req->visitType === 'other' || $req->visitType === 'general') {
                    $communicators = company_user::where('department', $req->departmentId)
                        ->where('designation', 'communicator')
                        ->where('user_type', 'employee')
                        ->get();

                    foreach ($communicators as $user) {
                        $tokenLink = URL::temporarySignedRoute(
                            'visit.autoLogin',
                            now()->addHours(24),
                            ['host_id' => $user->id, 'visit_id' => $visit->external_id]
                        );

                        if ($user->email) {
                            $this->mail_content = [
                                'subject' => "New General Visit Request",
                                'short_description' => "You have a new general visit request",
                                'content' => "A visitor <strong>{$visitor->first_name} {$visitor->last_name}</strong> has requested a general visit.<br>" .
                                             "Purpose: <strong>{$visit->purpose}</strong><br>" .
                                             "Requested on: " . now()->format('d M Y H:i A'),
                                'link' => $tokenLink,
                                'link2' => $tokenLink
                            ];

                            $users = [$user->id];
                            $session_data = session()->all();

                            dispatch(function () use ($session_data, $users) {
                                $this->mail_users = $this->get_users_by_id($this->mail_content, $users, [], ['session_data' => $session_data]);
                                $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
                            })->delay(now()->addSeconds(config("app.queue_time")));
                        }

                        $hostsData[] = [
                            'name' => trim($user->first_name . ' ' . $user->last_name),
                            'designation' => $user->designation ?? 'N/A',
                            'department' => $user->get_department
                                ? ['name' => $user->get_department->name]
                                : ['name' => 'N/A'],
                        ];
                    }
                }

              if ($req->visitType != 'other' && !empty($req->hostIds)) {
    foreach ($req->hostIds as $hostId) {
        $hostId = is_array($hostId) ? ($hostId['id'] ?? null) : $hostId;

        if (!$hostId) continue;

        $visitHost = new visit_host();
        $visitHost->visit_id = $visit->id;
        $visitHost->host_id = $hostId;
        $visitHost->save();

        $host = company_user::find($hostId);
        if ($host) {
            $v_host .= $host->first_name . ",";

            $hostsData[] = [
                'name' => trim($host->first_name . ' ' . $host->last_name),
                'designation' => $host->designation ?? 'N/A',
                'department' => $host->get_department
                    ? ['name' => $host->get_department->name]
                    : ['name' => 'N/A']
            ];

            if ($host->email) {
                $tokenLink = URL::temporarySignedRoute(
                    'visit.autoLogin',
                    now()->addHours(24),
                    ['host_id' => $host->id, 'visit_id' => $visit->external_id]
                );

                $this->mail_content = [
                    'subject' => "New Visit Request",
                    'short_description' => "You have a new visit request",
                    'content' => "You have received a new visit request from <strong>{$visitor->first_name} {$visitor->last_name}</strong>.<br>" .
                                 "Purpose: <strong>{$visit->purpose}</strong><br>" .
                                 "Requested on: " . now()->format('d M Y H:i A'),
                    'link' => $tokenLink,
                    'link2' => $tokenLink
                ];

                $users = [$host->id];
                $session_data = session()->all();

                dispatch(function () use ($session_data, $users) {
                    $this->mail_users = $this->get_users_by_id($this->mail_content, $users, [], ['session_data' => $session_data]);
                    $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
                })->delay(now()->addSeconds(config("app.queue_time")));
            }
        }
    }
    }

                if ($visitor->email) {
                    $this->mail_content = [
                        'subject' => "Your Host Has Been Notified",
                        'short_description' => "Your host has been informed about your visit",
                        'content' => "Hi {$visitor->first_name},<br><br>" .
                                     "Your <strong>host has been notified</strong> about your visit.<br>" .
                                     "We’ll keep you updated via email once your host responds.<br><br>" .
                                     "Thank you for visiting!",
                        'link' => '',
                        'link2' => ''
                    ];

                    $visitor_ids = [$visitor->id];
                    $session_data = session()->all();

                    dispatch(function () use ($session_data, $visitor_ids) {
                        $this->mail_visitors = $this->get_visitors_by_id($this->mail_content, $visitor_ids, [], ['session_data' => $session_data]);
                        $this->send_to_visitors($this->mail_visitors, ['session_data' => $session_data]);
                    })->delay(now()->addSeconds(config("app.queue_time")));
                }

                if ($req->customFields && is_array($req->customFields)) {
                    foreach ($req->customFields as $field) {
                        $value = is_array($field['value']) ? json_encode($field['value']) : $field['value'];

                        $customValue = new visitor_custom_field_value();
                        $customValue->visitor_id = $visitor->id;
                        $customValue->visit_id = $visit->id;
                        $customValue->custom_field_id = $field['id'];
                        $customValue->field_value = $value;
                        $customValue->save();
                    }
                }

                Generic::logActivity(
                    'visitor',
                    'check_in',
                    "Visitor {$visitor->first_name} {$visitor->last_name} checked in at {$visitor->checkin_time} to meet Host {$v_host}.",
                    $visitor->company_id
                );

                $code = $visitor->external_id . '_' . $visit->external_id;
                $qr_code = base64_encode(QrCode::format('png')->size(300)->generate($code));

                if ($req->ajax()) {
                    $response = [
                        'success' => true,
                        'message' => 'Please wait for your host’s response!',
                        'visitor' => [
                            'name' => $visitor->first_name . ' ' . $visitor->last_name,
                            'email' => $visitor->email,
                            'phone' => $visitor->phone,
                            'purpose' => $visitor->purpose_of_visit,
                            'id' => $visitor->external_id,
                            'visit_type' => $req->visitType,
                            'photo' => asset('storage/' . $visitor->photo_path),
                            'qr_code' => 'data:image/png;base64,' . $qr_code,
                            'code_value' => $code,
                            'checkin_time' => $visit->checkin_time ?? now()->format('Y-m-d H:i:s'),
                            'hosts' => $hostsData
                        ]
                    ];

                    return response()->json($response);
                }

                return redirect()->back()->with('success', 'Please wait for your host’s response!');
            });

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback explicitly for safety
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong: ' . $e->getMessage(),
            ], 500);
        }
    }
}

public function autoLogin(Request $request, $host_id, $visit_id)
{
    if (!$request->hasValidSignature()) {
        abort(401, 'Invalid or expired link.');
    }
    $user = company_user::find($host_id);
    if (!$user) {
        abort(404, 'Host not found.');
    }
    session([
        'comp_user_id' => $user->id,
        'comp_user_name' => trim($user->first_name . ' ' . $user->last_name),
        'company_id' => $user->company_id,
    ]);
  return redirect(url("/visit-profile/{$visit_id}"));
}

public function respond(Request $request)
{
    $validated = $request->validate([
        'visit_id'         => 'required|exists:visits,id',
        'status'           => 'required|in:Pending,Approved,Rejected',
        'response_message' => 'nullable|string|max:1000',
        'place_id'         => 'required',
        'duration'         => 'nullable',
        'start_time'       => 'required|date_format:H:i',
        'custom_duration'  => 'nullable|integer|min:5',
    ]);

$duration = $validated['duration'] === 'other'
    ? $validated['custom_duration']
    : $validated['duration'];

$duration = (int)($duration ?? 15);

$hostId = session('comp_user_id');
$startTime = Carbon::parse($validated['start_time']);
$endTime = $startTime->copy()->addMinutes($duration);
$conflict = visit_host_response::where('host_id', $hostId)
    ->get()
    ->contains(function ($existing) use ($startTime, $endTime) {
        if (!$existing->start_time || !$existing->duration) return false;

        $existingStart = Carbon::parse($existing->start_time);
        $existingEnd = $existingStart->copy()->addMinutes($existing->duration);
        return $startTime->lt($existingEnd) && $endTime->gt($existingStart);
    });

if ($conflict) {
    return response()->json([
        'success' => false,
        'message' => 'Time conflict detected — another visit is already scheduled for this time.',
    ], 409);
}

    $existingResponse = visit_host_response::where('visit_id', $validated['visit_id'])
        ->where('host_id', $hostId)
        ->first();

    if ($existingResponse) {
        $existingResponse->status           = $validated['status'];
        $existingResponse->response_message = $validated['response_message'];
        $existingResponse->place_id         = $validated['place_id'];
        $existingResponse->duration         = $duration ?? 15;
        $existingResponse->start_time       = $validated['start_time'];
        $existingResponse->approval_time    = now();
        $existingResponse->save();

        $response = $existingResponse;
    } else {
        $response = new visit_host_response();
        $response->visit_id       = $validated['visit_id'];
        $response->host_id        = $hostId;
        $response->status         = $validated['status'];
        $response->response_message = $validated['response_message'];
        $response->place_id       = $validated['place_id'];
        $response->duration       = $duration ?? 15;
        $response->start_time     = $validated['start_time'];
        $response->approval_time  = now();
        $response->save();
    }
     $visit = visit::find($validated['visit_id']);
    if ($visit) {
        $visit->status = "pending";
        $visit->save();
    }
     $host = company_user::find($hostId);
    $visitor = $visit ? $visit->get_visitor : null;
 if ($visitor && $visitor->email) {
    $this->mail_content['subject'] = "Visit Response Update";
    $this->mail_content['short_description'] = "Your Visit Request Has Been set as " . ucfirst($validated['status']);
    $this->mail_content['content'] = "Dear {$visitor->first_name},<br>"
        . "Your visit request has been set as <b>" . ucfirst($validated['status']) . "</b> by host "
        . "<b>" . ($host->first_name ?? 'N/A') . " " . ($host->last_name ?? '') . "</b>.<br>"
        . "Message: " . e($validated['response_message']) . "<br>"
        . "Place: " . ($response->get_visit_place->name ?? 'N/A') . "<br>"
        . "Duration: {$duration} minutes<br>"
        . "Time: {$validated['start_time']}<br>"
        . "Response recorded at " . now()->format("d M Y H:i A") . ".";
          $this->mail_content['link']  = '';
    $this->mail_content['link2'] = '';
    $this->mail_visitors = [];
    $visitors = [$visitor->id];
    $session_data = session()->all();

    dispatch(function () use ($session_data, $visitors) {
        $this->mail_visitors = $this->get_visitors_by_id(
            $this->mail_content,
            $visitors,
            $this->mail_visitors,
            ['session_data' => $session_data]
        );
        $this->send_to_visitors($this->mail_visitors, ['session_data' => $session_data]);
    })->delay(now()->addSeconds(config("app.queue_time")));
}

    return response()->json([
        'success' => true,
        'message' => 'Response recorded successfully.',
        'data'    => $response,
    ]);
}


public function profile($id)
{
    $visit = visit::with([
        'get_visitor',
        'get_host',
        'get_department',
        'get_visit_host_response.get_visit_place',
        'get_custom_field_values.get_custom_field'
    ])->where('external_id', $id)->first();

    if (!$visit) {
        return response()->json([
            'success' => false,
            'message' => 'Visit not found.'
        ], 404);
    }
    $visitorId = $visit->get_visitor?->external_id ?? 'UNKNOWN';
    $code = $visitorId . '_' . $visit->external_id;
    $qr_code = base64_encode(QrCode::format('png')->size(300)->generate($code));
    $visit->qr_image ='data:image/png;base64,' . $qr_code;

    return response()->json([
        'success' => true,
        'data' => $visit,
    ]);
}
public function check_visitor(Request $request)
{
    $email = $request->query('email');
    $firstName = $request->query('first_name');
    $lastName = $request->query('last_name');
    $phone = $request->query('phone');

    $query = visitor::query();
    $conditions = [];
    if ($email) {
        $conditions[] = ['email', '=', $email];
    }
    if ($firstName) {
        $conditions[] = ['first_name', 'LIKE', "%{$firstName}%"];
    }
    if ($lastName) {
        $conditions[] = ['last_name', 'LIKE', "%{$lastName}%"];
    }
    if ($phone) {
        $conditions[] = ['phone', 'LIKE', "%{$phone}%"];
    }
    if (empty($conditions)) {
        return response()->json(['exists' => false, 'message' => 'No search parameters provided']);
    }
    $record = visitor::where(function ($q) use ($conditions) {
        $count = count($conditions);
        for ($i = 0; $i < $count; $i++) {
            for ($j = $i + 1; $j < $count; $j++) {
                $q->orWhere(function ($sub) use ($conditions, $i, $j) {
                    $sub->where([$conditions[$i]])->where([$conditions[$j]]);
                });
            }
        }
    })->first();

    if ($record) {
        return response()->json([
            'exists' => true,
            'record' => $record
        ]);
    }

    return response()->json(['exists' => false]);
}

public function show($code)
{
    try {
        if (!str_contains($code, '_')) {
            return response()->json(['error' => 'Invalid QR code format'], 400);
        }
        [$visitorId, $visitId] = explode('_', $code, 2);
        if (empty($visitorId) || empty($visitId)) {
            return response()->json(['error' => 'Invalid QR code structure'], 400);
        }
        // return $visitorId;
        $visitor = visitor::where('external_id', $visitorId)->first();

        if (!$visitor) {
            return response()->json(['error' => 'Visitor not found'], 404);
        }
        // return $visitId;
       $visit = visit::where('external_id', $visitId)
            ->where('visitor_id', $visitor->id)
            ->where(function ($query) {
                $query->where('status', '!=', 'checkedout')
                    ->orWhereNull('checkout_time');
            })
            ->with('get_host', 'get_department')
            ->first();

        if (!$visit) {
            return response()->json(['error' => 'Visit not found or already checked out'], 404);
        }
        $visit->status = 'checkedout';
        $visit->checkout_time = now();
        if ($visit->checkin_time) {
            $checkin = \Carbon\Carbon::parse($visit->checkin_time);
            $checkout = \Carbon\Carbon::parse($visit->checkout_time);
            $durationInMinutes = $checkout->diffInMinutes($checkin);

            $hours = floor($durationInMinutes / 60);
            $minutes = $durationInMinutes % 60;

            $visit->duration = sprintf('%02dh %02dm', $hours, $minutes);
        } else {
            $visit->duration = null;
        }

        $visit->save();
        $hosts = $visit->get_host;
        $responses = visit_host_response::where('visit_id', $visit->id)
            ->with('get_visit_place')
            ->get();

        foreach ($hosts as $host) {
            $host->response = $responses->where('host_id', $host->id)->values();
        }

        return response()->json([
            'visitor' => $visitor,
            'visit' => $visit,
            'hosts' => $hosts,
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'error' => 'Server error',
            'message' => $e->getMessage(),
        ], 500);
    }
}
public function allCompanyVisits(Request $request)
{
    $employee = company_user::find(session('comp_user_id'));
    $company_id = $employee->company_id ?? session('company_id');

    $query = visit::with(['get_department', 'get_host', 'get_visitor'])
        ->where('company_id', $company_id);

    if ($request->filled('start_date')) {
        $query->whereDate('created_at', date('Y-m-d', strtotime($request->start_date)));
    }

    if ($request->filled('status')) {
        if ($request->status === 'checked-in') {
            $query->whereNull('checkout_time');
        } elseif ($request->status === 'checked-out') {
            $query->whereNotNull('checkout_time');
        }
    }

    if ($request->boolean('active_only')) {
        $query->whereNull('checkout_time');
    }

    if ($request->filled('host_id')) {
        $query->whereHas('get_host', function ($q) use ($request) {
            $q->where('host_id', $request->host_id);
        });
    }

    if ($request->filled('department_id')) {
        $query->where('department_id', $request->department_id);
    }

    $query->orderBy('created_at', 'desc');

    $paginate = filter_var($request->query('paginate', true), FILTER_VALIDATE_BOOLEAN);

    if ($paginate) {
        $perPage = $request->query('per_page', config('app.pagination', 10));
        $visits = $query->paginate($perPage);
        return response()->json([
            'success' => true,
            'visits' => $visits->items(),
            'pagination' => [
                'current_page' => $visits->currentPage(),
                'last_page' => $visits->lastPage(),
                'per_page' => $visits->perPage(),
                'total' => $visits->total(),
            ],
            'filters' => [
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'status' => $request->status,
                'active_only' => $request->boolean('active_only'),
                'host_id' => $request->host_id,
                'department_id' => $request->department_id,
            ],
        ]);
    } else {
        $visits = $query->get();
        return response()->json([
            'success' => true,
            'visits' => $visits,
            'filters' => [
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'status' => $request->status,
                'active_only' => $request->boolean('active_only'),
                'host_id' => $request->host_id,
                'department_id' => $request->department_id,
            ],
        ]);
    }
}


}
