<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\company;
use App\Models\custom_field;
use App\Models\custom_field_group;
use Illuminate\Support\Str;
use App\Http\Classes\Generic;
class CustomFieldController extends Controller
{
      public function index()
    {
        $fields = custom_field::where('company_id',session('company_id'))
            ->orderBy('id', 'desc')
            ->get();

        return response()->json(['fields' => $fields]);
    }
       public function list()
    {
        $fields = custom_field::whereNull('company_id')
            ->orderBy('order_no')
            ->get();

        return response()->json(['fields' => $fields]);
    }
    public function show($id)
    {
        $field = custom_field::findOrFail($id);
        return response()->json(['field' => $field]);
    }
public function store(Request $request)
{
    // return $request;
    $request->validate([
        'group_id' => 'required|integer|exists:custom_field_groups,id',
        'fields' => 'required|array',
        'fields.*.field_label' => 'required|string|max:255',
        'fields.*.field_type' => 'required|in:text,number,email,date,time,textarea,select,checkbox,radio,file',
        'fields.*.is_required' => 'nullable|boolean',
        'fields.*.order_no' => 'nullable|integer',
        'fields.*.status' => 'nullable|in:active,inactive',
    ]);

    $savedFields = [];

    foreach ($request->fields as $fieldData) {
        $field = new custom_field();
        $field->external_id = Str::uuid();
        $field->company_id = session('company_id');
        $field->group_id = $request->group_id;
        $field->field_name = Str::slug($fieldData['field_label'], '_');
        $field->field_label = $fieldData['field_label'];
        $field->field_type = $fieldData['field_type'];
        $field->is_required = $fieldData['is_required'] ?? false;
        $field->order_no = $fieldData['order_no'] ?? 0;
        $field->status = $fieldData['status'] ?? 'active';
        $field->save();

        $savedFields[] = $field;
    }

    return response()->json([
        'message' => 'Custom fields created successfully.',
        'fields' => $savedFields
    ]);
}


    public function update(Request $request, $id)
    {
        $field = custom_field::findOrFail($id);

        $request->validate([
            'field_name' => 'required|string|max:255',
            'field_label' => 'required|string|max:255',
            'field_type' => 'required|in:text,number,email,date,time,textarea,select,checkbox,radio,file',
            'is_required' => 'boolean',
            'order_no' => 'integer',
            'status' => 'in:active,inactive',
        ]);

        $field->update($request->all());

        return response()->json(['message' => 'Custom field updated successfully', 'field' => $field]);
    }
    public function destroy($id)
    {
        $field = custom_field::findOrFail($id);
        $field->delete();

        return response()->json(['message' => 'Custom field deleted successfully']);
    }
public function groupList(Request $request)
{
    $search = $request->query('search');
    $externalId = $request->query('external_id');
    $paginate = $request->query('paginate', false);
    $companyId = session('company_id');

    if (!$companyId && $externalId) {
        $company = company::where('external_id', $externalId)->first();
        if ($company) {
            $companyId = $company->id;
        }
    }

    if (!$companyId) {
        return response()->json([
            'message' => 'Company not found.',
            'groups' => [],
        ], 404);
    }

    $query = custom_field_group::where('company_id', $companyId)
        ->with('get_fields')
        ->orderBy('id', 'desc');

    if (!empty($search)) {
        $query->where('name', 'like', "%{$search}%");
    }

    if ($paginate) {
        // Return paginated results
        $groups = $query->paginate(config('app.pagination', 10));
        return response()->json([
            'groups' => $groups->items(),
            'pagination' => [
                'current_page' => $groups->currentPage(),
                'last_page' => $groups->lastPage(),
                'per_page' => $groups->perPage(),
                'total' => $groups->total(),
            ],
            'message' => 'Custom Field Groups fetched successfully (paginated).',
        ]);
    } else {
        // Return all results without pagination
        $groups = $query->get();
        return response()->json([
            'groups' => $groups,
            'message' => 'All Custom Field Groups fetched successfully.',
        ]);
    }
}




    public function groupStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:custom_field_groups,name',
        ]);

        $group = new  custom_field_group();
        $group->name = $validated['name'];
        $group->company_id= session('company_id');
        $group->save();

        return response()->json([
            'message' => 'Custom Field Group created successfully.',
            'group' => $group
        ]);
    }

    public function groupDestroy($id)
    {
        $group = custom_field_group::find($id);

        if (!$group) {
            return response()->json([
                'message' => 'Custom Field Group not found.'
            ], 404);
        }

        $group->delete();

        return response()->json([
            'message' => 'Custom Field Group deleted successfully.'
        ]);
    }
 public function getGroupFields($groupId)
    {
        $group = custom_field_group::find($groupId);
        if (!$group) {
            return response()->json(['message' => 'Group not found'], 404);
        }
        $fields = custom_field::where('group_id', $groupId)->get();

        return response()->json($fields);
    }

}
