<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\company;
use App\Models\custom_field;
use App\Models\custom_field_table_value;
use App\Models\custom_field_table_column;
use App\Models\custom_field_group;
use Illuminate\Support\Str;
use App\Http\Classes\Generic;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CustomFieldsImport;

class CustomFieldController extends Controller
{
      public function index()
    {
        $fields = custom_field::where('company_id',session('company_id'))
            ->orderBy('id', 'asc')
            ->get();

        return response()->json(['fields' => $fields]);
    }
       public function list()
    {
        $fields = custom_field::whereNull('company_id')
            ->orderBy('order_no')
            ->get();

        return response()->json(['fields' => $fields]);
    }
    public function show($id)
    {
        $field = custom_field::findOrFail($id);
        return response()->json(['field' => $field]);
    }
public function store(Request $request)
{
    // return $request;
    $request->validate([
        'group_id' => 'required|exists:custom_field_groups,external_id',
        'fields' => 'required|array',
        'fields.*.field_label' => 'required|string',
        'fields.*.field_type' => 'required|in:text,number,email,date,time,textarea,select,checkbox,radio,file',
        'fields.*.is_required' => 'nullable|boolean',
        'fields.*.order_no' => 'nullable|integer',
        'fields.*.status' => 'nullable|in:active,inactive',
    ]);

    $savedFields = [];
 $group = custom_field_group::where('external_id',$request->group_id)->firstOrFail();
    foreach ($request->fields as $fieldData) {
        $field = new custom_field();
        $field->external_id = Generic::generate_id("cf");
        $field->company_id = session('company_id');
        $field->group_id = $group->id;
        $field->field_name = Str::slug($fieldData['field_label'], '_');
        $field->field_label = $fieldData['field_label'];
        $field->field_type = $fieldData['field_type'];
        $field->is_required = $fieldData['is_required'] ?? false;
        $field->order_no = $fieldData['order_no'] ?? 0;
        $field->status = $fieldData['status'] ?? 'active';
        $field->save();

        $savedFields[] = $field;
    }

    return response()->json([
        'message' => 'Custom fields created successfully.',
        'fields' => $savedFields
    ]);
}


    public function update(Request $request, $id)
    {
        $field = custom_field::findOrFail($id);

        $request->validate([
            'field_name' => 'required|string',
            'field_label' => 'required|string',
            'field_type' => 'required|in:text,number,email,date,time,textarea,select,checkbox,radio,file',
            'is_required' => 'boolean',
            'order_no' => 'integer',
            'status' => 'in:active,inactive',
        ]);

        $field->update($request->all());

        return response()->json(['message' => 'Custom field updated successfully', 'field' => $field]);
    }
    public function destroy($id)
    {
        $field = custom_field::findOrFail($id);
        $field->delete();

        return response()->json(['message' => 'Custom field deleted successfully']);
    }
 public function groupList(Request $request)
{
    $search = $request->query('search');
    $externalId = $request->query('external_id');
    $paginate = $request->query('paginate', false);
    $companyId = session('company_id');

    if (!$companyId && $externalId) {
        $company = company::where('external_id', $externalId)->first();
        if ($company) {
            $companyId = $company->id;
        }
    }

    if (!$companyId) {
        return response()->json([
            'message' => 'Company not found.',
            'groups' => [],
        ], 404);
    }

    $query = custom_field_group::where('company_id', $companyId)
        ->with(['get_fields' => function($q) {
            $q->whereNull('parent_id')
              ->with(['get_subFields.get_table_columns', 'get_table_columns']);
        }])
        ->orderBy('id', 'asc');

    if (!empty($search)) {
        $query->where('name', 'like', "%{$search}%");
    }

    if ($paginate) {
        $groups = $query->paginate(config('app.pagination', 10));
        return response()->json([
            'groups' => $groups->items(),
            'pagination' => [
                'current_page' => $groups->currentPage(),
                'last_page' => $groups->lastPage(),
                'per_page' => $groups->perPage(),
                'total' => $groups->total(),
            ],
            'message' => 'Custom Field Groups fetched successfully (paginated).',
        ]);
    } else {
        $groups = $query->get();
        return response()->json([
            'groups' => $groups,
            'message' => 'All Custom Field Groups fetched successfully.',
        ]);
    }
}

    public function groupStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string',
        ]);

        $group = new  custom_field_group();
        $group->name = $validated['name'];
        $group->external_id= Generic::generate_id("grp");
        $group->company_id= session('company_id');
        $group->save();

        return response()->json([
            'message' => 'Custom Field Group created successfully.',
            'group' => $group
        ]);
    }

    public function groupDestroy($id)
    {
        $group = custom_field_group::find($id);

        if (!$group) {
            return response()->json([
                'message' => 'Custom Field Group not found.'
            ], 404);
        }

        $group->delete();

        return response()->json([
            'message' => 'Custom Field Group deleted successfully.'
        ]);
    }
 public function getGroupFields($groupId)
    {
        $group = custom_field_group::where('external_id',$groupId)->firstOrFail();
        if (!$group) {
            return response()->json(['message' => 'Group not found'], 404);
        }
         $fields = custom_field::with(['get_subFields.get_table_columns','get_table_columns'])->where('group_id', $group->id)->whereNull('parent_id')->get();

        return response()->json($fields);
    }


    public function storeSubFields(Request $request, $parentId)
    {
        $parent = custom_field::find($parentId);
        if (!$parent) {
            return response()->json(['message' => 'Parent field not found'], 404);
        }
        $request->validate([
            'subFields' => 'required|array',
            'subFields.*.label' => 'required|string',
            'subFields.*.field_type' => 'required',
            'subFields.*.is_required' => 'nullable|boolean',
            'subFields.*.display_on' => 'required|in:yes,no,always',
        ]);

        $savedSubFields = [];

        foreach ($request->subFields as $subData) {
            $subField = new custom_field();
            $subField->external_id = Str::uuid();
            $subField->company_id = session('company_id');
            $subField->group_id = $parent->group_id;
            $subField->parent_id = $parent->id;
            $subField->field_name = Str::slug($subData['label'], '_');
            $subField->field_label = $subData['label'];
            $subField->field_type = $subData['field_type'];
            $subField->is_required = $subData['is_required'] ?? false;
            $subField->display_on = $subData['display_on'];
            $subField->order_no = $subData['order_no'] ?? 0;
            $subField->status = $subData['status'] ?? 'active';
            $subField->save();
            if ($subField->field_type === 'table' && isset($subData['columns'])) {
            foreach ($subData['columns'] as $colData) {
                $column = new custom_field_table_column();
                $column->custom_field_id = $subField->id;
                $column->name = $colData['label'];
                $column->order_no = $colData['order_no'] ?? 0;
                $column->save();
            }
        }
            $savedSubFields[] = $subField;
        }

        return response()->json([
            'message' => 'Sub-fields created successfully.',
            'subFields' => $savedSubFields
        ]);
    }

 public function import(Request $request)
{
    if ($request->isMethod("POST")) {
        if ($request->hasFile('file')) {
            $file = $request->file("file");
            $path1 = $file->store('temp');
            $path = storage_path('app') . '/' . $path1;
        } else {
            return response()->json([
                'status' => 'fail',
                'message' => "File not found or not supported"
            ], 422);
        }

        $requiredColumns = ['group_id', 'label', 'type', 'is_required'];

        try {
            $data = Excel::toArray(new CustomFieldsImport, $path)[0];
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'fail',
                'message' => "Failed to read file: " . $e->getMessage()
            ], 422);
        }
        foreach ($requiredColumns as $required) {
            if (!array_key_exists($required, $data[0])) {
                return response()->json([
                    'status' => 'fail',
                    'message' => "$required column is required"
                ], 422);
            }
        }

        $rowCounter = 0;
        $messages = [];

                foreach ($data as $row) {
                    try {
                        $rowCounter++;
                        $missing = [];

                        foreach ($requiredColumns as $field) {
                            if (!isset($row[$field]) || $row[$field] === "") {
                                $missing[] = $field;
                            }
                        }

                        if ($missing) {
                            $messages[] = (object)[
                                "row_counter" => $rowCounter,
                                "status" => "fail",
                                "group_id" => $row['group_id'] ?? '',
                                "group_name" => '',
                                "label" => $row['label'] ?? '',
                                "type" => $row['type'] ?? '',
                                "is_required" => $row['is_required'] ?? '',
                                "message" => "Required fields missing: " . implode(", ", $missing)
                            ];
                            continue;
                        }

                        $validTypes = ['text', 'email', 'number', 'radio'];
                        if (!in_array(strtolower($row['type']), $validTypes)) {
                            $messages[] = (object)[
                                "row_counter" => $rowCounter,
                                "status" => "fail",
                                "group_name" => '',
                                "label" => $row['label'],
                                "message" => "Invalid field type [" . $row['type'] . "]"
                            ];
                            continue;
                        }

                        $externalGroupId = trim($row['group_id']);
                        $group = custom_field_group::where('external_id', $externalGroupId)->first();

                        if (!$group) {
                            $messages[] = (object)[
                                "row_counter" => $rowCounter,
                                "status" => "fail",
                                "group_id" => $externalGroupId,
                                "group_name" => '',
                                "label" => $row['label'],
                                "message" => "Custom Field Group ID [" . $externalGroupId . "] does not exist"
                            ];
                            continue;
                        }

                        // FIXED HERE
                        $existingField = custom_field::where('group_id', $group->id)
                            ->where('field_label', $row['label'])
                            ->first();

                        if ($existingField) {
                            $messages[] = (object)[
                                "row_counter" => $rowCounter,
                                "status" => "fail",
                                "group_id" => $externalGroupId,
                                "group_name" => $group->name,
                                "label" => $row['label'],
                                "message" => "Custom Field '{$row['label']}' already exists in this group"
                            ];
                            continue;
                        }

                        // SAVE NEW FIELD
                        $slug = Str::slug($row['label'], '-');
                        $field = new custom_field();
                        $field->group_id = $group->id;
                        $field->field_label = $row['label'];
                        $field->field_name = $slug;
                        $field->field_type = strtolower($row['type']);
                        $field->is_required = (int)$row['is_required'];
                        $field->external_id = Generic::generate_id('custom-field');
                        $field->company_id = session('company_id');
                        $field->save();

                        $messages[] = (object)[
                            "row_counter" => $rowCounter,
                            "status" => "success",
                            "group_id" => $externalGroupId,
                            "group_name" => $group->name,
                            "label" => $row['label'],
                            "type" => $row['type'],
                            "is_required" => $row['is_required'],
                            "message" => "Custom Field '{$row['label']}' created successfully"
                        ];

                    } catch (\Exception $e) {
                        $messages[] = (object)[
                            "row_counter" => $rowCounter,
                            "status" => "fail",
                            "label" => $row['label'] ?? 'N/A',
                            "message" => "Error: " . $e->getMessage()
                        ];
                    }
                }



        return response()->json([
            'status' => 'success',
            'messages' => $messages
        ]);
    }

    return response()->json([
        'status' => 'fail',
        'messages' => []
    ]);
}


}
