<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Mail;

use Illuminate\Http\Request;
use App\Models\custom_field;
use App\Models\visitor;
use Carbon\Carbon;
use App\Models\company_instruction;
use App\Models\notification;
use App\Models\visit;
use App\Models\activity_log;
use App\Mail\invitation_mail;
use App\Models\visitor_custom_field_value;
use Illuminate\Support\Str;
use App\Http\Classes\Generic;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\company_user;
use App\Models\visit_host;
use App\Models\visit_host_response;
use App\Http\Traits\SystemMail;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\DB;
use App\Models\custom_field_table_value;
use App\Models\custom_field_table_column;

class VisitController extends Controller
{
    use SystemMail;
    public function store(Request $req)
    {
        // return $req;
        if ($req->isMethod('post')) {
            $validator = Validator::make($req->all(), [
                'firstName' => 'required|string|max:255',
                'lastName' => 'required|string|max:255',
                'company' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'phone' => 'required|string|max:20',
                'city' => 'nullable|string|max:100',
                'state' => 'nullable|string|max:100',
                'country' => 'nullable|string|max:100',
                 'idPhoto' => 'required|string',
                'purpose' => 'required|string|max:150',
                'customFields' => 'array',
                'customFields.*.id' => 'required|integer',
                'schedule' => 'required|string',
            ]);

            if ($validator->fails()) {
                if ($req->ajax()) {
                    return response()->json(['errors' => $validator->errors()], 422);
                }
                return redirect()->back()->withErrors($validator)->withInput();
            }

            try {
                return DB::transaction(function () use ($req) {
                    $v_host = '';
                    $visitor = visitor::where('email', $req->email)->first();

                    if ($visitor) {
                        $visitor->first_name = $req->firstName;
                        $visitor->last_name = $req->lastName;
                        $visitor->organization = $req->company;
                        $visitor->phone = $req->phone;
                        $visitor->whatsapp = $req->whatsapp ?? $visitor->whatsapp;
                        $visitor->city = $req->city;
                        $visitor->state = $req->state;
                        $visitor->country = $req->country;
                        $visitor->address = $req->address ?? $visitor->address;
                        $visitor->blacklisted = 'false';
                    } else {
                        $visitor = new visitor();
                        $visitor->external_id = strtoupper(Str::random(6));
                        $visitor->first_name = $req->firstName;
                        $visitor->last_name = $req->lastName;
                        $visitor->company_id = session('company_id');
                        $visitor->organization = $req->company;
                        $visitor->email = $req->email;
                        $visitor->phone = $req->phone;
                        $visitor->whatsapp = $req->whatsapp ?? null;
                        $visitor->city = $req->city;
                        $visitor->state = $req->state;
                        $visitor->country = $req->country;
                        $visitor->blacklisted = 'false';
                        $visitor->address = $req->address ?? null;
                    }

                    if ($req->idPhoto) {
                        $imageData = str_replace('data:image/png;base64,', '', $req->idPhoto);
                        $imageData = str_replace(' ', '+', $imageData);
                        $imageName = 'visitor_' . time() . '.png';
                        $imagePath = storage_path('app/public/visitors/' . $imageName);
                        file_put_contents($imagePath, base64_decode($imageData));
                        $visitor->photo_path = 'visitors/' . $imageName;
                    }
                    if ($req->signature) {
                    $sigData = str_replace('data:image/png;base64,', '', $req->signature);
                    $sigData = str_replace(' ', '+', $sigData);
                    $sigName = 'signature_' . time() . '.png';
                    $sigPath = storage_path('app/public/visitors/' . $sigName);
                    file_put_contents($sigPath, base64_decode($sigData));
                    $visitor->signature = 'visitors/' . $sigName; // column from migration
                }
                    $visitor->purpose_of_visit = $req->purpose;
                    $visitor->save();

                    $visit = new visit();
                    $visit->visitor_id = $visitor->id;
                    $visit->external_id = strtoupper(Str::random(6));
                    $visit->department_id = $req->departmentId;
                    $visit->visit_type = $req->visitType;
                    $visit->purpose = $req->purpose;
                    $visit->status = 'checkedin';
                    $visit->checkin_time = now();
                    $visit->visit_date = now();
                    $visit->company_id = session('company_id');
                    $visit->scheduled = $req->schedule ?? 'Unscheduled';
                    $visit->save();

                    $hostsData = [];
                    $code = $visitor->external_id . '_' . $visit->external_id;
                    $qr_code = base64_encode(QrCode::format('png')->size(300)->generate($code));

                    if ($req->visitType === 'other' || $req->visitType === 'general') {
                        $communicators = company_user::where('department', $req->departmentId)->where('designation', 'communicator')->where('user_type', 'employee')->get();

                        foreach ($communicators as $user) {
                            $tokenLink = URL::temporarySignedRoute('visit.autoLogin', now()->addHours(24), ['host_id' => $user->id, 'visit_id' => $visit->external_id]);

                            if ($user->email) {
                                $this->mail_content = [
                                    'subject' => 'New General Visit Request',
                                    'short_description' => 'You have a new general visit request',
                                    'content' => "A visitor <strong>{$visitor->first_name} {$visitor->last_name}</strong> has requested a general visit.<br>" . "Purpose: <strong>{$visit->purpose}</strong><br>" . 'Requested on: ' . now()->format('d M Y H:i A'),
                                    'link' => $tokenLink,
                                    'link2' => $tokenLink,
                                ];

                                $users = [$user->id];
                                $session_data = session()->all();

                                dispatch(function () use ($session_data, $users) {
                                    $this->mail_users = $this->get_users_by_id($this->mail_content, $users, [], ['session_data' => $session_data]);
                                    $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
                                })->delay(now()->addSeconds(config('app.queue_time')));
                            }

                            $hostsData[] = [
                                'name' => trim($user->first_name . ' ' . $user->last_name),
                                'designation' => $user->designation ?? 'N/A',
                                'department' => $user->get_department ? ['name' => $user->get_department->name] : ['name' => 'N/A'],
                            ];
                        }
                    }

                    if ($req->visitType != 'other' && !empty($req->hostIds)) {
                        foreach ($req->hostIds as $hostId) {
                            $hostId = is_array($hostId) ? $hostId['id'] ?? null : $hostId;

                            if (!$hostId) {
                                continue;
                            }

                            $visitHost = new visit_host();
                            $visitHost->visit_id = $visit->id;
                            $visitHost->host_id = $hostId;
                            $visitHost->save();

                            $host = company_user::find($hostId);
                            if ($host) {
                                $v_host .= $host->first_name . ',';

                                $hostsData[] = [
                                    'name' => trim($host->first_name . ' ' . $host->last_name),
                                    'designation' => $host->designation ?? 'N/A',
                                    'department' => $host->get_department ? ['name' => $host->get_department->name] : ['name' => 'N/A'],
                                ];

                                if ($host->email) {
                                    $tokenLink = URL::temporarySignedRoute('visit.autoLogin', now()->addHours(24), ['host_id' => $host->id, 'visit_id' => $visit->external_id]);

                                    $this->mail_content = [
                                        'subject' => 'New Visit Request',
                                        'short_description' => 'You have a new visit request',
                                        'content' => "You have received a new visit request from <strong>{$visitor->first_name} {$visitor->last_name}</strong>.<br>" . "Purpose: <strong>{$visit->purpose}</strong><br>" . 'Requested on: ' . now()->format('d M Y H:i A'),
                                        'link' => $tokenLink,
                                        'link2' => $tokenLink,
                                    ];

                                    $users = [$host->id];
                                    $session_data = session()->all();

                                    dispatch(function () use ($session_data, $users) {
                                        $this->mail_users = $this->get_users_by_id($this->mail_content, $users, [], ['session_data' => $session_data]);
                                        $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
                                    })->delay(now()->addSeconds(config('app.queue_time')));
                                }
                            }
                        }
                    }

                    if ($visitor->email) {
                        $this->mail_content = [
                            'subject' => 'Your Host Has Been Notified',
                            'short_description' => 'Your host has been informed about your visit',
                            'content' => "Hi {$visitor->first_name},<br><br>" . 'Your <strong>host has been notified</strong> about your visit.<br>' . 'We’ll keep you updated via email once your host responds.<br><br>' . 'Thank you for visiting!',
                            'link' => '',
                            'link2' => '',
                        ];

                        $visitor_ids = [$visitor->id];
                        $session_data = session()->all();

                        dispatch(function () use ($session_data, $visitor_ids) {
                            $this->mail_visitors = $this->get_visitors_by_id($this->mail_content, $visitor_ids, [], ['session_data' => $session_data]);
                            $this->send_to_visitors($this->mail_visitors, ['session_data' => $session_data]);
                        })->delay(now()->addSeconds(config('app.queue_time')));
                    }

                    if ($req->customFields && is_array($req->customFields)) {
                        foreach ($req->customFields as $field) {
                              $dbField = custom_field::find($field['id']);
                                        if ($dbField && $dbField->field_type === 'table') {

                                            if (is_array($field['value'])) {

                                                foreach ($field['value'] as $rowIndex => $row) {

                                                    $rowId = $rowIndex + 1;

                                                    foreach ($row as $columnName => $cellValue) {
                                                        $column = custom_field_table_column::where('custom_field_id', $dbField->id)
                                                            ->where('name', $columnName)
                                                            ->first();

                                                        if (!$column) continue;

                                                        $tableValue = new custom_field_table_value();
                                                        $tableValue->visitor_id = $visitor->id;
                                                        $tableValue->visit_id = $visit->id;
                                                        $tableValue->custom_field_id = $dbField->id;
                                                        $tableValue->row_id = $rowId;
                                                        $tableValue->column_id = $column->id;
                                                        $tableValue->value = $cellValue;
                                                        $tableValue->save();
                                                    }
                                                }
                                            }

                                            continue;
                                        }

                            $value = is_array($field['value']) ? json_encode($field['value']) : $field['value'];

                            $customValue = new visitor_custom_field_value();
                            $customValue->visitor_id = $visitor->id;
                            $customValue->visit_id = $visit->id;
                            $customValue->custom_field_id = $field['id'];
                            $customValue->field_value = $value;
                            $customValue->save();
                        }
                    }

                    Generic::logActivity('visitor', 'check_in', "Visitor {$visitor->first_name} {$visitor->last_name} checked in at {$visit->checkin_time} to meet Host {$v_host}.", $visit->company_id, $visit->id);

                    if ($req->ajax()) {
                        $response = [
                            'success' => true,
                            'message' => 'Please wait for your host’s response!',
                            'visitor' => [
                                'name' => $visitor->first_name . ' ' . $visitor->last_name,
                                'email' => $visitor->email,
                                'phone' => $visitor->phone,
                                'purpose' => $visitor->purpose_of_visit,
                                'id' => $visitor->external_id,
                                'visit_type' => $req->visitType,
                                'photo' => asset('storage/' . $visitor->photo_path),
                                'qr_code' => 'data:image/png;base64,' . $qr_code,
                                'code_value' => $code,
                                'checkin_time' => $visit->checkin_time ?? now()->format('Y-m-d H:i:s'),
                                'hosts' => $hostsData,
                            ],
                        ];

                        return response()->json($response);
                    }

                    return redirect()->back()->with('success', 'Please wait for your host’s response!');
                });
            } catch (\Exception $e) {
                DB::rollBack(); // Rollback explicitly for safety
                return response()->json(
                    [
                        'success' => false,
                        'message' => 'Something went wrong: ' . $e->getMessage(),
                    ],
                    500,
                );
            }
        }
    }

    public function autoLogin(Request $request, $host_id, $visit_id)
    {
        if (!$request->hasValidSignature()) {
            abort(401, 'Invalid or expired link.');
        }
        $user = company_user::find($host_id);
        if (!$user) {
            abort(404, 'Host not found.');
        }
        session([
            'comp_user_id' => $user->id,
            'comp_user_name' => trim($user->first_name . ' ' . $user->last_name),
            'company_id' => $user->company_id,
        ]);
        return redirect(url("/visit-profile/{$visit_id}"));
    }

    public function respond(Request $request)
    {
        $validated = $request->validate([
            'visit_id' => 'required|exists:visits,id',
            'status' => 'required|in:Pending,Approved,Rejected',
            'response_message' => 'nullable|string|max:1000',
            'place_id' => 'required_if:status,Approved',
            'duration' => 'required_if:status,Approved',
            'start_time' => 'required_if:status,Approved|date_format:H:i',
            'custom_duration' => 'nullable|integer|min:5|required_if:duration,other',
        ]);

        $duration = $validated['duration'] === 'other' ? $validated['custom_duration'] : $validated['duration'];

        $duration = (int) ($duration ?? 15);

        $hostId = session('comp_user_id');
        $startTime = Carbon::today()->setTimeFromTimeString($validated['start_time']);
        $endTime = $startTime->copy()->addMinutes($duration);
        if ($validated['status'] === 'Approved') {
            $startTime = Carbon::today()->setTimeFromTimeString($validated['start_time']);
            $endTime = $startTime->copy()->addMinutes($duration);
            $existingResponses = visit_host_response::where('visit_id', $validated['visit_id'])->where('status', 'Approved')->where('host_id', '!=', $hostId)->get();

            $conflict = $existingResponses->contains(function ($response) use ($startTime, $endTime) {
                if (!$response->start_time || !$response->duration) {
                    return false;
                }
                $existingStart = Carbon::today()->setTimeFromTimeString($response->start_time);
                $existingEnd = $existingStart->copy()->addMinutes((int) $response->duration);
                return $startTime->lt($existingEnd) && $endTime->gt($existingStart);
            });

            if ($conflict) {
                return response()->json(
                    [
                        'success' => false,
                        'message' => 'Time conflict detected — another host has already scheduled this visit during this time.',
                    ],
                    409,
                );
            }
        }

        $existingResponse = visit_host_response::where('visit_id', $validated['visit_id'])->where('host_id', $hostId)->first();

        if ($existingResponse) {
            $existingResponse->status = $validated['status'];
            $existingResponse->response_message = $validated['response_message'];

            if ($validated['status'] === 'Approved') {
                $existingResponse->place_id = $validated['place_id'];
                $existingResponse->duration = $duration ?? 15;
                $existingResponse->start_time = $validated['start_time'];
                $existingResponse->approval_time = now();
            } else {
                $existingResponse->place_id = null;
                $existingResponse->duration = null;
                $existingResponse->start_time = null;
                $existingResponse->approval_time = null;
            }

            $existingResponse->save();
            $response = $existingResponse;
        } else {
            $response = new visit_host_response();
            $response->visit_id = $validated['visit_id'];
            $response->host_id = $hostId;
            $response->status = $validated['status'];
            $response->response_message = $validated['response_message'];

            if ($validated['status'] === 'Approved') {
                $response->place_id = $validated['place_id'];
                $response->duration = $duration ?? 15;
                $response->start_time = $validated['start_time'];
                $response->approval_time = now();
            } else {
                $response->place_id = null;
                $response->duration = null;
                $response->start_time = null;
                $response->approval_time = null;
            }

            $response->save();
        }

        $visit = visit::find($validated['visit_id']);
        if ($visit) {
            $visit->status = 'pending';
            $visit->save();
        }
        $visitor = $visit ? $visit->get_visitor : null;
        $host = company_user::find($hostId);

        if ($visitor && $visitor->email) {
            $statusText = ucfirst($validated['status']);
            $this->mail_content['subject'] = 'Visit Response Update';
            $this->mail_content['short_description'] = 'Update on Your Visit Request';

            $baseMessage = "Dear {$visitor->first_name},<br>" . "Your visit request has been <b>{$statusText}</b> by " . '<b>' . ($host->first_name ?? 'N/A') . ' ' . ($host->last_name ?? '') . '</b>.<br>' . "Host's Message: " . e($validated['response_message']) . '<br>';

            if (strtolower($validated['status']) === 'approved') {
                $baseMessage .= 'Place: ' . ($response->get_visit_place->name ?? 'N/A') . '<br>' . "Duration: {$duration} minutes<br>" . "Time: {$validated['start_time']}<br>";
            }

            $baseMessage .= 'Response recorded at ' . now()->format('d M Y H:i A') . '.';

            $this->mail_content['content'] = $baseMessage;
            $this->mail_content['link'] = '';
            $this->mail_content['link2'] = '';
            $this->mail_visitors = [];
            $visitors = [$visitor->id];
            $session_data = session()->all();

            dispatch(function () use ($session_data, $visitors) {
                $this->mail_visitors = $this->get_visitors_by_id($this->mail_content, $visitors, $this->mail_visitors, ['session_data' => $session_data]);
                $this->send_to_visitors($this->mail_visitors, ['session_data' => $session_data]);
            })->delay(now()->addSeconds(config('app.queue_time')));


                Generic::logActivity(
                    'visitor',
                    $statusText, // approved / rejected
                    "Visit request for {$visitor->first_name} {$visitor->last_name} was {$statusText} by {$host->first_name} {$host->last_name}.",
                    $visitor->company_id,
                    $visit->id
                );
        }

        return response()->json([
            'success' => true,
            'message' => 'Response recorded successfully.',
            'data' => $response,
        ]);
    }
public function profile($id)
{
    $hostId = session('comp_user_id');

    $visit = visit::with([
        'get_visitor',
        'get_host',
        'get_department',
        'get_custom_field_values' => function ($query) {
            $query->with(['get_custom_field']);
        },
        'get_custom_field_values.get_custom_field.get_group',
        'get_custom_field_values.get_custom_field.get_subFields' => function ($query) use ($id) {
            $query->with([
                'get_table_columns' => function ($colQuery) use ($id) {
                    $colQuery->with([
                        'get_table_values' => function ($valQuery) use ($id) {
                            $valQuery->where('visit_id', function ($q) use ($id) {
                                $q->select('id')
                                  ->from('visits')
                                  ->where('external_id', $id)
                                  ->limit(1);
                            });
                        }
                    ]);
                }
            ]);
        },
        'get_visit_host_response' => function ($query) use ($hostId) {
            $query->where('host_id', $hostId)->with('get_visit_place');
        },
    ])
    ->where('external_id', $id)
    ->first();

    if (!$visit) {
        return response()->json([
            'success' => false,
            'message' => 'Visit not found.',
        ], 404);
    }

    $visitorId = $visit->get_visitor?->external_id ?? 'UNKNOWN';
    $code = $visitorId . '_' . $visit->external_id;
    $qr_code = base64_encode(QrCode::format('png')->size(300)->generate($code));
    $visit->qr_image = 'data:image/png;base64,' . $qr_code;

    $current_user = company_user::find($hostId);

    return response()->json([
        'success' => true,
        'data' => $visit,
        'currentUser' => [
            'id' => $current_user->id,
            'role' => $current_user->designation,
        ],
    ]);
}


    public function check_visitor(Request $request)
    {
        $email = $request->query('email');
        $firstName = $request->query('first_name');
        $lastName = $request->query('last_name');
        $phone = $request->query('phone');

        $query = visitor::query();
        $conditions = [];
        if ($email) {
            $conditions[] = ['email', '=', $email];
        }
        if ($firstName) {
            $conditions[] = ['first_name', 'LIKE', "%{$firstName}%"];
        }
        if ($lastName) {
            $conditions[] = ['last_name', 'LIKE', "%{$lastName}%"];
        }
        if ($phone) {
            $conditions[] = ['phone', 'LIKE', "%{$phone}%"];
        }
        if (empty($conditions)) {
            return response()->json(['exists' => false, 'message' => 'No search parameters provided']);
        }
        $record = visitor::where(function ($q) use ($conditions) {
            $count = count($conditions);
            for ($i = 0; $i < $count; $i++) {
                for ($j = $i + 1; $j < $count; $j++) {
                    $q->orWhere(function ($sub) use ($conditions, $i, $j) {
                        $sub->where([$conditions[$i]])->where([$conditions[$j]]);
                    });
                }
            }
        })
            ->where('company_id', session('company_id'))
            ->first();

        if ($record) {
            return response()->json([
                'exists' => true,
                'record' => $record,
            ]);
        }

        return response()->json(['exists' => false]);
    }

   public function show($code)
{
    try {
        if (!str_contains($code, '_')) {
            return response()->json(['error' => 'Invalid QR code format'], 400);
        }

        [$visitorId, $visitId] = explode('_', $code, 2);

        if (empty($visitorId) || empty($visitId)) {
            return response()->json(['error' => 'Invalid QR code structure'], 400);
        }

        $visitor = visitor::where('external_id', $visitorId)->first();
        if (!$visitor) {
            return response()->json(['error' => 'Visitor not found'], 404);
        }

        $visit = visit::where('external_id', $visitId)
            ->where('visitor_id', $visitor->id)
            ->where(function ($query) {
                $query->where('status', '!=', 'checkedout')
                      ->orWhereNull('checkout_time');
            })
            ->with('get_host', 'get_department')
            ->first();

        if (!$visit) {
            return response()->json(['error' => 'Visit not found or already checked out'], 404);
        }

        // --- Checkout process ---
        $visit->status = 'checkedout';
        $visit->checkout_time = now();

        if ($visit->checkin_time) {
            $checkin = \Carbon\Carbon::parse($visit->checkin_time);
            $checkout = \Carbon\Carbon::parse($visit->checkout_time);
            $durationInMinutes = $checkout->diffInMinutes($checkin);
            $hours = floor($durationInMinutes / 60);
            $minutes = $durationInMinutes % 60;
            $visit->duration = sprintf('%02dh %02dm', $hours, $minutes);
        } else {
            $visit->duration = null;
        }

        $visit->save();

        $hosts = $visit->get_host;
        $responses = visit_host_response::where('visit_id', $visit->id)
            ->with('get_visit_place')
            ->get();

        foreach ($hosts as $host) {
            $host->response = $responses->where('host_id', $host->id)->values();
        }

        // --- Send emails after checkout ---
        $session_data = session()->all();

        // Email to Visitor
        if (!empty($visitor->email)) {
            $this->mail_content = [
                'subject' => "Thank You for Visiting",
                'short_description' => "Your visit has been successfully checked out",
                'content' => "Hi {$visitor->first_name},<br><br>" .
                    "We hope you had a great visit! You have been successfully <strong>checked out</strong>.<br>" .
                    "Visit duration: <strong>{$visit->duration}</strong><br><br>" .
                    "Thank you for visiting us!",
                'link' => '',
                'link2' => ''
            ];

            $visitor_ids = [$visitor->id];
            dispatch(function () use ($session_data, $visitor_ids) {
                $this->mail_visitors = $this->get_visitors_by_id($this->mail_content, $visitor_ids, [], ['session_data' => $session_data]);
                $this->send_to_visitors($this->mail_visitors, ['session_data' => $session_data]);
            })->delay(now()->addSeconds(config("app.queue_time")));
        }

        // Email to all hosts
        if (!empty($hosts)) {
            $this->mail_content = [
            'subject' => "Visitor Checked Out",
                'short_description' => "A visitor has checked out",
                'content' => "Hello,<br><br>" .
                    "Your visitor <strong>{$visitor->first_name} {$visitor->last_name}</strong> has just <strong>checked out</strong>.<br>" .
                    "Visit duration: <strong>{$visit->duration}</strong><br><br>" .
                    "Thank you for hosting!",
                'link' => config("app.domain") . "auth/admin-login",
                'link2' => config("app.domain") . "auth/admin-login",
            ];


            $users = $hosts->pluck('id')->toArray();

              dispatch(function () use ($session_data, $users) {
            $this->mail_users = $this->get_users_by_id($this->mail_content, $users, $this->mail_users, ['session_data' => $session_data]);
            $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
        })->delay(now()->addSeconds(config("app.queue_time")));
        }

        return response()->json([
            'visitor' => $visitor,
            'visit' => $visit,
            'hosts' => $hosts,
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'error' => 'Server error',
            'message' => $e->getMessage(),
        ], 500);
    }
}

    public function allCompanyVisits(Request $request)
    {
        $employee = company_user::find(session('comp_user_id'));
        $company_id = $employee->company_id ?? session('company_id');

        $query = visit::with(['get_department', 'get_host', 'get_visitor'])->where('company_id', $company_id);

        if ($request->filled('start_date')) {
            $query->whereDate('created_at', date('Y-m-d', strtotime($request->start_date)));
        }

        if ($request->filled('status')) {
            if ($request->status === 'checked-in') {
                $query->whereNull('checkout_time');
            } elseif ($request->status === 'checked-out') {
                $query->whereNotNull('checkout_time');
            }
        }

        if ($request->boolean('active_only')) {
            $query->whereNull('checkout_time');
        }

        if ($request->filled('host_id')) {
            $query->whereHas('get_host', function ($q) use ($request) {
                $q->where('host_id', $request->host_id);
            });
        }

        if ($request->filled('department_id')) {
            $query->where('department_id', $request->department_id);
        }

        $query->orderBy('created_at', 'desc');

        $paginate = filter_var($request->query('paginate', true), FILTER_VALIDATE_BOOLEAN);

        if ($paginate) {
            $perPage = $request->query('per_page', config('app.pagination', 10));
            $visits = $query->paginate($perPage);
            return response()->json([
                'success' => true,
                'visits' => $visits->items(),
                'pagination' => [
                    'current_page' => $visits->currentPage(),
                    'last_page' => $visits->lastPage(),
                    'per_page' => $visits->perPage(),
                    'total' => $visits->total(),
                ],
                'filters' => [
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'status' => $request->status,
                    'active_only' => $request->boolean('active_only'),
                    'host_id' => $request->host_id,
                    'department_id' => $request->department_id,
                ],
            ]);
        } else {
            $visits = $query->get();
            return response()->json([
                'success' => true,
                'visits' => $visits,
                'filters' => [
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'status' => $request->status,
                    'active_only' => $request->boolean('active_only'),
                    'host_id' => $request->host_id,
                    'department_id' => $request->department_id,
                ],
            ]);
        }
    }
public function add_instruction(Request $req)
{
    if ($req->isMethod('post')) {
        $validator = Validator::make($req->all(), [
            'instruction' => 'required|string',
        ]);

        if ($validator->fails()) {
            if ($req->ajax()) {
                return response()->json(['errors' => $validator->errors()], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $instruction = new company_instruction();
        $instruction->company_id = session('company_id');
        $instruction->instructions = $req->instruction;
        $instruction->save();
        if ($req->ajax()) {
            return response()->json(['message' => 'Instruction created successfully!']);
        }
        return redirect()->back()->with('success', 'Instruction created successfully!');
    }

    return view('admin.instructions.create');
}
public function instruction_list(Request $request)
{
    $search = $request->query('search');
    $externalId = $request->query('external_id');
    $paginate = $request->query('paginate', false);
    $companyId = session('company_id');

    $perPage = config('app.pagination', 10);
    if (!$companyId && $externalId) {
        $company = company::where('external_id', $externalId)->first();
        if ($company) {
            $companyId = $company->id;
        }
    }

    if (!$companyId) {
        return response()->json([
            'message' => 'Company not found.',
            'instructions' => [],
        ], 404);
    }
    $query = company_instruction::where('company_id', $companyId)
        ->orderBy('id', 'desc');
    if (!empty($search)) {
        $query->where('instructions', 'LIKE', "%{$search}%");
    }
    if ($paginate) {
        $instructions = $query->paginate($perPage, ['id', 'instructions','created_at']);

        return response()->json([
            'instructions' => $instructions->items(),
            'pagination' => [
                'current_page' => $instructions->currentPage(),
                'last_page' => $instructions->lastPage(),
                'per_page' => $instructions->perPage(),
                'total' => $instructions->total(),
            ],
            'message' => 'Instructions fetched successfully (paginated).',
        ]);
    }
    $instructions = $query->get(['id', 'instructions', 'created_at']);

    return response()->json([
        'instructions' => $instructions,
        'message' => 'All instructions fetched successfully.',
    ]);
}
public function delete_instruction(Request $request, $id)
{
    $instruction = company_instruction::find($id);

    if (!$instruction) {
        if ($request->ajax()) {
            return response()->json(['message' => 'Instruction not found.'], 404);
        }
        return redirect()->back()->with('error', 'Instruction not found.');
    }
    $instruction->delete();

    if ($request->ajax()) {
        return response()->json(['message' => 'Instruction deleted successfully!']);
    }

    return redirect()->back()->with('success', 'Instruction deleted successfully!');
}

}
