<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Models\department;
use App\Models\company;
use App\Models\visit;
use App\Http\Classes\Generic;
use App\Models\company_user;
use App\Http\Traits\SystemMail;
use App\Models\visitor_custom_field_value;
use Illuminate\Validation\Rule;
use App\Models\activity_log;
class EmployeeController extends Controller
{
        use SystemMail;
 public function add(Request $req)
    {
        if ($req->isMethod('post')) {
            $validator = Validator::make($req->all(), [
                'first_name'    => 'required|string|max:255',
                'last_name'     => 'required|string|max:255',
                'email'         => 'required|email|max:255|unique:company_users,email',
                'phone'         => 'nullable|string|max:20',
                'state'         => 'nullable|string|max:100',
                'country'       => 'nullable|string|max:100',
                'city'          => 'nullable|string|max:100',
                'address'       => 'nullable|string|max:255',
                'postal_code'   => 'nullable|string|max:20',
                'designation'   => 'nullable|string|max:255',
                'department_id' => 'required|exists:departments,id',
                'status'        => 'required|in:active,inactive',
            ]);

            if ($validator->fails()) {
                if ($req->ajax()) {
                    return response()->json(['errors' => $validator->errors()], 422);
                }
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $employee = new company_user();
            $employee->external_id  = Generic::generate_id('emp');
            $employee->company_id   = session('company_id');
            $employee->first_name   = $req->first_name;
            $employee->last_name    = $req->last_name;
            $employee->email        = $req->email;
            $employee->phone        = $req->phone;
            $employee->state        = $req->state;
            $employee->country      = $req->country;
            $employee->city         = $req->city;
            $employee->address      = $req->address;
            $employee->postal_code  = $req->postal_code;
            $employee->designation  = $req->designation;
            $employee->department = $req->department_id;
            $employee->status       = $req->status;
            $employee->user_type    = 'employee';

            $password = Str::upper(Str::random(6));
            $employee->password     = $password;
            $employee->save();
        $this->mail_content['subject'] = "Account Created";
        $this->mail_content['short_description'] = "Welcome!" ;
        $this->mail_content['link'] = "login";
        $this->mail_content['content'] = "Your account has been successfully created at "
            . date("d M Y H:i A")
            . "<br>Login Details: <br>Email: " . $employee->email
            . "<br>Password: " . $employee->password;
        $this->mail_content['link2'] = config("app.domain") . "auth/admin-login";

        $this->mail_users = [];
        $users = [$employee->id];
        $session_data = session()->all();
        dispatch(function () use ($session_data, $users) {
            $this->mail_users = $this->get_users_by_id($this->mail_content, $users, $this->mail_users, ['session_data' => $session_data]);
            $this->send_to_members($this->mail_users, ['session_data' => $session_data]);
        })->delay(now()->addSeconds(config("app.queue_time")));
            if ($req->ajax()) {
                return response()->json(['message' => 'Employee created successfully!']);
            }

            return redirect()->back()->with('success', 'Employee created successfully!');
        }

        return view('admin.employees.create');
    }

public function list(Request $request)
{
    $search = $request->get('search');
    $externalId = $request->query('external_id');
    $paginate = $request->query('paginate', false); // check if pagination is requested
    $companyId = session('company_id');

    if (!$companyId && $externalId) {
        $company = company::where('external_id', $externalId)->first();
        if ($company) {
            $companyId = $company->id;
        }
    }

    if (!$companyId) {
        return response()->json(['employees' => []], 404);
    }

    $query = company_user::with('get_department')
        ->select('id', 'external_id', 'first_name', 'last_name', 'email', 'designation', 'department', 'status')
        ->where('company_id', $companyId)
        ->when($search, function ($query, $search) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$search}%"])
                  ->orWhere('email', 'like', "%{$search}%");
            });
        })
        ->orderBy('id', 'desc');

    if ($paginate) {
        $perPage = $request->query('per_page', config('app.pagination', 10));
        $employees = $query->paginate($perPage);

        $employees->getCollection()->transform(function ($emp) {
            $emp->department_name = $emp->get_department->name ?? '-';
            return $emp;
        });

        return response()->json([
            'employees' => $employees->items(),
            'pagination' => [
                'total' => $employees->total(),
                'per_page' => $employees->perPage(),
                'current_page' => $employees->currentPage(),
                'last_page' => $employees->lastPage(),
            ],
        ]);
    } else {
        $employees = $query->get();

        $employees->transform(function ($emp) {
            $emp->department_name = $emp->get_department->name ?? '-';
            return $emp;
        });

        return response()->json([
            'employees' => $employees,
            'message' => 'All employees fetched successfully.',
        ]);
    }
}

public function delete($id)
{
    $employee = company_user::find($id);
    if (!$employee) {
        return response()->json(['message' => 'Employee not found.'], 404);
    }

    $employee->delete();

    return response()->json(['message' => 'Employee deleted successfully!']);
}
public function show($id)
{
    $employee = company_user::with('get_department:id,name')
        ->where('external_id',$id)->first();

    return response()->json(['employee' => $employee]);
}

public function update(Request $req, $id)
{
    $employee = company_user::where('external_id',$id)->first();

    $validator = Validator::make($req->all(), [
       'first_name'    => 'required|string|max:255',
                'last_name'     => 'required|string|max:255',
                'email'         => [
                'required',
                'email',
                'max:255',
                Rule::unique('company_users')->ignore($employee->id),
            ],
                'phone'         => 'nullable|string|max:20',
                'state'         => 'nullable|string|max:100',
                'country'       => 'nullable|string|max:100',
                'city'          => 'nullable|string|max:100',
                'address'       => 'nullable|string|max:255',
                'postal_code'   => 'nullable|string|max:20',
                'designation'   => 'nullable|string|max:255',
                'department_id' => 'required|exists:departments,id',
                'status'        => 'required|in:active,inactive',
    ]);

    if ($validator->fails()) {
        return response()->json(['errors' => $validator->errors()], 422);
    }
    $employee->first_name    = $req->first_name;
    $employee->last_name     = $req->last_name;
    $employee->email         = $req->email;
    $employee->phone         = $req->phone;
    $employee->state         = $req->state;
    $employee->country       = $req->country;
    $employee->city          = $req->city;
    $employee->address       = $req->address;
    $employee->postal_code   = $req->postal_code;
    $employee->designation   = $req->designation ?? '';
    $employee->department = $req->department_id;
    $employee->status        = $req->status;

    $employee->save();

    return response()->json(['message' => 'Employee updated successfully']);
}

public function myVisits(Request $request)
{
    $employee = company_user::find(session('comp_user_id'));
    $perPage = $request->query('per_page', config('app.pagination', 10));
    $page = $request->query('page', 1);
    $status = $request->query('status');
    $startDate = $request->query('start_date');

    $query = visit::with(['get_department', 'get_host', 'get_visitor', 'get_custom_field_values'])
        ->where(function ($query) use ($employee) {
            if (strtolower($employee->designation) === 'communicator') {
                $query->where(function ($q) use ($employee) {
                    $q->where(function ($q2) use ($employee) {
                        $q2->whereIn('visit_type', ['general', 'other'])
                           ->where('department_id', $employee->department);
                    })
                    ->orWhereHas('get_host', function ($q3) use ($employee) {
                        $q3->where('company_users.id', $employee->id);
                    });
                });
            } else {
                $query->whereHas('get_host', function ($q) use ($employee) {
                    $q->where('company_users.id', $employee->id);
                });
            }
        })
        ->orderBy('created_at', 'desc');

    // ✅ Apply filters
    if ($status) {
        $query->where('status', $status);
    }

    if ($startDate) {
        $query->whereDate('created_at', $startDate);
    }

    $query->orderBy('created_at', 'desc');

    $visits = $query->paginate($perPage);

    return response()->json([
        'visits' => $visits->items(),
        'pagination' => [
            'current_page' => $visits->currentPage(),
            'last_page' => $visits->lastPage(),
            'per_page' => $visits->perPage(),
            'total' => $visits->total(),
        ],
    ]);
}

public function my_profile()
{
    $user = company_user::with('get_department:id,name')
        ->select('id', 'first_name', 'last_name', 'email', 'designation', 'department', 'status', 'company_id', 'phone', 'address', 'city', 'state', 'country')
        ->where('id', session('comp_user_id'))
        ->where('company_id', session('company_id'))
        ->first();

    if (!$user) {
        return response()->json(['error' => 'User not found'], 404);
    }

    return response()->json([
        'user' => [
            'name'       => trim($user->first_name . ' ' . $user->last_name),
            'email'      => $user->email,
            'phone'      => $user->phone ?? '-',
            'department' => $user->get_department->name ?? '-',
            'designation'=> $user->designation ?? '-',
            'address'    => $user->address ?? '-',
            'city'       => $user->city ?? '-',
            'state'      => $user->state ?? '-',
            'country'    => $user->country ?? '-',
        ]
    ]);
}

    public function dashboard()
    {
        $company_id = session('company_id');

        //  Basic Counts
        $total_visits = visit::where('company_id', $company_id)->count();
        $today_visits = visit::where('company_id', $company_id)->whereDate('created_at', date('Y-m-d'))->count();
        $current_visits = visit::where('company_id', $company_id)->whereDate('created_at', date('Y-m-d'))->whereNull('checkout_time')->count();
        $checked_out_visits = visit::where('company_id', $company_id)->whereDate('created_at', date('Y-m-d'))->whereNotNull('checkout_time')->count();

        // Recent Activity Logs
        $activity_logs = activity_Log::with('get_visit')
            ->where('company_id', $company_id)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($log) {
                return [
                    'id' => $log->id,
                    'message' => $log->message ?? 'No message',
                    'time' => $log->created_at->format('h:i A'),
                ];
            });

        // 👥 Recent Visitors (today)
        $recent_visitors = visit::with('get_visitor', 'get_department', 'get_host:id,first_name,last_name')->where('company_id', $company_id)->whereDate('created_at', date('Y-m-d'))->whereNull('checkout_time')->orderBy('id', 'desc')->limit(10)->get();

        $visitorsOverTime = [];

        //  Visits by Department
        // $departmentsData = Visit::selectRaw('department_id, COUNT(*) as total')->where('company_id', $company_id)->whereDate('created_at', date('Y-m-d'))->groupBy('department_id')->with('get_department')->get();

        // $departmentLabels = [];
        // $departmentValues = [];
        // foreach ($departmentsData as $dept) {
        //     $departmentLabels[] = $dept->get_department->name ?? 'Unknown';
        //     $departmentValues[] = $dept->total;
        // }

        return response()->json([
            'totals' => [
                'total' => $total_visits,
                'today' => $today_visits,
                'current' => $current_visits,
                'checked_out' => $checked_out_visits,
            ],

            // 'departments' => [
            //     'labels' => $departmentLabels,
            //     'values' => $departmentValues,
            // ],
            'recentVisitors' => $recent_visitors,
            'logs' => $activity_logs,
        ]);
    }

    public function getHosts()
    {
        $company_id = session('company_id'); // static company id

        $hosts = company_user::where('company_id', $company_id)
            ->select('id', 'first_name', 'last_name', 'email', 'phone')
            ->get()
            ->map(function ($host) {
                return [
                    'id' => $host->id,
                    'name' => trim(($host->first_name ?? '') . ' ' . ($host->last_name ?? '')),
                    'email' => $host->email ?? '-',
                    'phone' => $host->phone ?? '-',
                ];
            });

        return response()->json(['hosts' => $hosts]);
    }


}
